/**
 * @projectDescription	i2b2 XLS Export Plugin 
 * @inherits			i2b2
 * @namespace			i2b2.ExportXLS
 * @author				Axel Newe
 * ----------------------------------------------------------------------------------------
 * updated 2013-01-09: Initial Launch [Axel Newe, FAU Erlangen-Nuremberg] 
 */

 i2b2.ExportXLS.DEBUG_GetPropertyList = function(object)
 {
	var propertyList = "";
	
	for (var thisPropertyName in object) {
		propertyList += thisPropertyName + '\n';
	}
 
	return propertyList;
 }
 
i2b2.ExportXLS.InitAndResetGlobalVars = function()
{
 	// Global var to store concept columns for "single_filtered" and "single_detailed" mode
	i2b2.ExportXLS.ConceptColumns = new Array();

 	// Global vars to store count, lower border & upper border for chunked PDO queries.
	i2b2.ExportXLS.PDOQueryPatientCount = 0;
	i2b2.ExportXLS.PDOQueryChunkLower = 0;
	i2b2.ExportXLS.PDOQueryChunkUpper = 0;
	i2b2.ExportXLS.PDOQueryChunkCurrent = 1;
	
	// Global var to flag whether query has been paged by server
	i2b2.ExportXLS.PDOQueryPagedByServer = false;
	
	// Global var to store result matrix
	i2b2.ExportXLS.ResultMatrix = new Array();

	// Global var to store CSV result
	i2b2.ExportXLS.CSVExport = "";

	// Global var to store HTML table result
	i2b2.ExportXLS.HTMLResult = "";

	// Global var to store all patient sets returned
	i2b2.ExportXLS.Patients = new Array();
	
	// Global var to store all observation sets returned
	i2b2.ExportXLS.ObservationSets = {};
	
	// Global var for patient demographic data column order. This also defines which columns are considered at all!
	// The full set yould be: 
	 i2b2.ExportXLS.PatientDataColumns = new Array ( "sex_cd", "age_in_years_num", "birth_date", "birth_year", "vital_status_cd", "language_cd", "marital_status_cd", "race_cd", "religion_cd", "income_cd", "statecityzip_path", "state_path", "city_path", "zip_cd")
	
	// Global var for patient demographic data column headers 
	i2b2.ExportXLS.PatientDataColumnHeaders = { "vital_status_cd": "Vital Status", "language_cd": "Language", "birth_date": "Birth Date", "birth_year": "Birth Year", "race_cd": "Race", "religion_cd": "Regligion", "income_cd": "Income", "statecityzip_path": "State/City/ZIP", "state_path": "State", "city_path": "City", "zip_cd": "ZIP", "marital_status_cd": "Marital Status", "age_in_years_num": "Age in Years", "sex_cd": "Sex" }	
	
}
 
i2b2.ExportXLS.ResetOptionsToDefault = function()
{
	// Default output options
	i2b2.ExportXLS.model.dirtyResultsData = true;
	i2b2.ExportXLS.model.dirtyMatrixData = true;
	i2b2.ExportXLS.model.outputOptions = {};	
	i2b2.ExportXLS.model.outputOptions.queryPageSize = 0;
	i2b2.ExportXLS.model.outputOptions.replacePatientID = false;
	i2b2.ExportXLS.model.outputOptions.excludeDelimiter = false;
	i2b2.ExportXLS.model.outputOptions.resolveConceptDetails = false;
	i2b2.ExportXLS.model.outputOptions.includeOntologyPath = false;
	i2b2.ExportXLS.model.outputOptions.outputFormat = "single_filtered";
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Sex = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Age = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_BirthDate = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_BirthYear = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_VitalStatus = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Language = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Race = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Religion = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Income = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_StateCityZIP = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_State = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_City = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_ZIP = false;
	i2b2.ExportXLS.model.outputOptions.includePatientDemoData_MaritalStatus = false;
} 

i2b2.ExportXLS.UpdateProgressMessage = function(progressMessage)
{
	$("results-working-progress").innerHTML = progressMessage;
	$("results-working-hard-progress").innerHTML = progressMessage;
}

i2b2.ExportXLS.CalcPDOChunkUpper = function() 
{
	var lower = i2b2.ExportXLS.PDOQueryChunkLower;
	var upper = i2b2.ExportXLS.PDOQueryPatientCount; // Set upper border by default to # of patients
	var total = i2b2.ExportXLS.PDOQueryPatientCount;

	var chunkSize = i2b2.ExportXLS.model.outputOptions.queryPageSize;
	
	if (chunkSize != 0)
	{
		upper = lower + chunkSize;
		
		if (upper > total)
		{
			upper = total;
		}		
	}

	i2b2.ExportXLS.PDOQueryChunkUpper = upper;
	
	return;
}

i2b2.ExportXLS.AllChunksQueried = function()
{
	result = false;
	
	if (i2b2.ExportXLS.PDOQueryChunkUpper == i2b2.ExportXLS.PDOQueryPatientCount)
	{
		result = true;	
	}

	return result;
}

i2b2.ExportXLS.ResultsTabSelected = function(ev)
{
	// Tabs have changed 
	if (ev.newValue.get('id')=="ExportXLS-TAB1") 
	{
		// User switched to Results table_html
		if (  (i2b2.ExportXLS.model.concepts.length > 0) && (i2b2.ExportXLS.model.prsRecord)  ) 
		{				
			if (i2b2.ExportXLS.model.dirtyResultsData) // Query new result data only if relevant input data has changed
			{				
				i2b2.ExportXLS.ClearResults();
				
				i2b2.ExportXLS.PDOQueryPagedByServer = false;

				i2b2.ExportXLS.PDOQueryPatientCount = i2b2.ExportXLS.model.prsRecord.origData.size;
				i2b2.ExportXLS.PDOQueryChunkLower = 0;
				i2b2.ExportXLS.CalcPDOChunkUpper();
				i2b2.ExportXLS.PDOQueryChunkCurrent = 0;
				
				i2b2.ExportXLS.GetResults(); 
			}
			else if (i2b2.ExportXLS.model.dirtyMatrixData) // Reformat matrix if necessary
			{				
				i2b2.ExportXLS.FinishProcessing();
			}
		}
	}
}

i2b2.ExportXLS.Init = function(loadedDiv) 
{
	i2b2.ExportXLS.debugCounter = 0;
	
	// Register DIV as valid drag&drop target for Patient Record Set (PRS) objects
	var op_trgt = {dropTarget:true}
	
	i2b2.sdx.Master.AttachType("ExportXLS-CONCPTDROP", "CONCPT", op_trgt);
	i2b2.sdx.Master.AttachType("ExportXLS-PRSDROP", "PRS", op_trgt);

	// Drop event handlers used by this plugin
	i2b2.sdx.Master.setHandlerCustom("ExportXLS-CONCPTDROP", "CONCPT", "DropHandler", i2b2.ExportXLS.ConceptDropped);
	i2b2.sdx.Master.setHandlerCustom("ExportXLS-PRSDROP", "PRS", "DropHandler", i2b2.ExportXLS.PatientRecordSetDropped);

	// Array to store concepts
	i2b2.ExportXLS.model.concepts = [];	
	i2b2.ExportXLS.ResetOptionsToDefault();	
	
	// Global var to cache denotation strings for concept codes
	i2b2.ExportXLS.ConceptCodeDenotations = {};

	// Global var to cache ontology paths of concept codes
	i2b2.ExportXLS.ConceptCodeOntologyPaths = {};

	// Init global vars
	i2b2.ExportXLS.InitAndResetGlobalVars();

	// Manage YUI tabs
	this.yuiTabs = new YAHOO.widget.TabView("ExportXLS-TABS", {activeIndex:0});
	this.yuiTabs.on('activeTabChange', function(ev) { i2b2.ExportXLS.ResultsTabSelected(ev) } );
	
	// Fix IE scrollbar problem (thanks to Wayne Chan...)
	var z = $('anaPluginViewFrame').getHeight() - 34;
	var mainContentDivs = $$('DIV#ExportXLS-TABS DIV.ExportXLS-MainContent');
	for (var i = 0; i < mainContentDivs.length; i++) 
	{
		mainContentDivs[i].style.height = z;
	}
}

i2b2.ExportXLS.Unload = function() 
{
	// Mop up old data
	i2b2.ExportXLS.model.prsRecord = false;
	i2b2.ExportXLS.model.conceptRecord = false;
	i2b2.ExportXLS.model.dirtyResultsData = true;
	
	i2b2.ExportXLS.ResetOptionsToDefault();
	i2b2.ExportXLS.InitAndResetGlobalVars();

	return true;
}

i2b2.ExportXLS.PatientRecordSetDropped = function(sdxData) 
{
	sdxData = sdxData[0];	// only interested in first record
	// save the info to our local data model
	i2b2.ExportXLS.model.prsRecord = sdxData;
	// let the user know that the drop was successful by displaying the name of the patient set
	$("ExportXLS-PRSDROP").innerHTML = i2b2.h.Escape(sdxData.sdxInfo.sdxDisplayName);
	// temporarly change background color to give GUI feedback of a successful drop occuring
	$("ExportXLS-PRSDROP").style.background = "#CFB";
	setTimeout("$('ExportXLS-PRSDROP').style.background='#DEEBEF'", 250);	
	// optimization to prevent requerying the hive for new results if the input dataset has not changed
	i2b2.ExportXLS.model.dirtyResultsData = true;		
}

i2b2.ExportXLS.ConceptDropped = function(sdxData) 
{
	sdxData = sdxData[0];	// Consider first record only 
		
	var conceptKeyValue = sdxData.sdxInfo.sdxKeyValue;  // Save the info to local data model

	if(i2b2.ExportXLS.model.concepts.length) 
	{	
		var duplicateConceptFound = false;
		
		for (var i3 = 0; i3 < i2b2.ExportXLS.model.concepts.length; i3++) 
		{		
			var conceptKeyValueToCompare = i2b2.ExportXLS.model.concepts[i3].sdxInfo.sdxKeyValue;
									
			if (conceptKeyValueToCompare == conceptKeyValue)
			{
				duplicateConceptFound = true;
			}
			else
			{
				// do nothing, duplicateConceptFound can stay false;
			}
		}
		
		if(!duplicateConceptFound)
		{		
			i2b2.ExportXLS.model.concepts.push(sdxData);			
			i2b2.ExportXLS.ConceptsRender();  				// Sort and display the concept list			
			i2b2.ExportXLS.model.dirtyResultsData = true;	// Optimization to prevent requerying the hive for new results if the input dataset has not changed
		}
		else 
		{
			alert("Impossible to drag a duplicate concept!");
		}	
	}
	else 
	{
		i2b2.ExportXLS.model.concepts.push(sdxData);		
		i2b2.ExportXLS.ConceptsRender(); 				// Sort and display the concept list		
		i2b2.ExportXLS.model.dirtyResultsData = true; 	// Optimization to prevent requerying the hive for new results if the input dataset has not changed
	}		
			
}

i2b2.ExportXLS.ConceptDelete = function(concptIndex) 
{
	i2b2.ExportXLS.model.concepts.splice(concptIndex,1); // remove the selected concept
	i2b2.ExportXLS.ConceptsRender(); 					 // sort and display the concept list
	i2b2.ExportXLS.model.dirtyResultsData = true;		 // optimization to prevent requerying the hive for new results if the input dataset has not changed
}

i2b2.ExportXLS.SetFlag = function(ckBox, option, dirtyResults) 
{
	if (Object.isUndefined(dirtyResults))
	{
		dirtyResults = true;
	}

	if (dirtyResults)  
	{
		i2b2.ExportXLS.model.dirtyResultsData = true;
	}	

	i2b2.ExportXLS.model.outputOptions[option] = ckBox.checked;
	i2b2.ExportXLS.model.dirtyMatrixData = true;
}

i2b2.ExportXLS.SetComboOption = function(comboBox, option, dirtyResults) 
{
	if (Object.isUndefined(dirtyResults))
	{
		dirtyResults = true;
	}

	if (dirtyResults)  
	{
		i2b2.ExportXLS.model.dirtyResultsData = true;
	}	

	i2b2.ExportXLS.model.outputOptions[option] = comboBox.options[comboBox.selectedIndex].value;
	i2b2.ExportXLS.model.dirtyMatrixData = true;
}

i2b2.ExportXLS.SetPositiveIntValue = function(editField, option, dirtyResults) 
{
	intValue = parseInt(editField.value);
	
	if (isNaN(intValue))
	{
		alert("Please enter an integer number!");
		editField.value = "0";
	}
	else if (intValue < 0)
	{
		alert("Please enter a positive integer number!");
		editField.value = "0";
	}
	else
	{
		i2b2.ExportXLS.model.outputOptions[option] = intValue;
		editField.value = intValue;
		
		if (dirtyResults)  
		{
			i2b2.ExportXLS.model.dirtyResultsData = true;
		}
		
		i2b2.ExportXLS.model.dirtyMatrixData = true;
	}
}

i2b2.ExportXLS.QueryResultHasErrors = function(resultData)
{
	if (resultData.error) 
	{
		alert('The results from the server could not be understood.\n\n' + resultData);
		console.error("Bad Results from Cell Communicator: ", resultData);
		return true;
	}
	
	return false;
}

i2b2.ExportXLS.GetXMLResponseFromQueryResult = function(resultData)
{
	var xmlContent; 
	var xmlDoc;
	
	if (window.DOMParser) // not Internet Explorer
	{ 
		xmlContent = resultData.msgResponse; // for Chrome, Safari, & FireFox 
		var parser = new DOMParser();
		xmlDoc = parser.parseFromString(xmlContent,"text/xml"); 
	} 
	else // Internet Explorer
	{ 
		xmlDoc = new ActiveXObject("Microsoft.XMLDOM");
		xmlDoc.async = false;
		xmlDoc.loadXML(resultData.msgResponse);
		xmlContent = xmlDoc; 
	}
	
	var result = new Array();
	result["xmlDoc"] = xmlDoc;
	result["xmlContent"] = xmlContent;
		
	return result;
}

i2b2.ExportXLS.ReturnCSVResult = function() 
{
	//i2b2.ExportXLS.CSVExport = "HEAD1;HEAD2;HEAD3\nDATA1;DATA2;DATA3\nDATA1;DATA2;DATA3";	
	return i2b2.ExportXLS.CSVExport;
}

i2b2.ExportXLS.ReturnHTMLResult = function() 
{
	var HTMLExport = "<html>\n<head></head>\n<body>\n" + i2b2.ExportXLS.HTMLResult + "\</body>\n</html>";

	return HTMLExport;
}

i2b2.ExportXLS.ConceptsRender = function()
{
	var s = '';
	
	if (i2b2.ExportXLS.model.concepts.length)  // If there are any concepts in the list
	{		
		// Sort the concepts in alphabetical order
		i2b2.ExportXLS.model.concepts.sort(function() {return arguments[0].sdxInfo.sdxDisplayName > arguments[1].sdxInfo.sdxDisplayName}); 
		
		// Draw the list of concepts
		for (var i1 = 0; i1 < i2b2.ExportXLS.model.concepts.length; i1++) 
		{
			if (i1 > 0) { s += '<div class="concptDiv"></div>'; }
			s += '<a class="concptItem" href="JavaScript:i2b2.ExportXLS.ConceptDelete(' + i1 + ');">' + i2b2.h.Escape(i2b2.ExportXLS.model.concepts[i1].sdxInfo.sdxDisplayName) + '</a>';
		}
		
		// Show the delete message
		$("ExportXLS-DeleteMsg").style.display = 'block';		
	} 
	else   // No concepts selected yet
	{
		s = '<div class="concptItem">Drop one or more Concepts here</div>';
		$("ExportXLS-DeleteMsg").style.display = 'none';
	}
	
	// Update html
	$("ExportXLS-CONCPTDROP").innerHTML = s;
}

i2b2.ExportXLS.GetPDOQueryFilter = function(min, max) 
{
	// Translate the concept XML for injection as PDO item XML 
	var filterList = '';
	
	for (var i1=0; i1<i2b2.ExportXLS.model.concepts.length; i1++)
	{
		var t = i2b2.ExportXLS.model.concepts[i1].origData.xmlOrig;
		var cdata = {};
		cdata.level = i2b2.h.getXNodeVal(t, "level");
		cdata.key = i2b2.h.getXNodeVal(t, "key");
		cdata.tablename = i2b2.h.getXNodeVal(t, "tablename");
		cdata.dimcode = i2b2.h.getXNodeVal(t, "dimcode");
		cdata.synonym = i2b2.h.getXNodeVal(t, "synonym_cd");
		filterList +=
		'	<panel name="' + cdata.key + '">\n' +
		'		<panel_number>0</panel_number>\n' +
		'		<panel_accuracy_scale>0</panel_accuracy_scale>\n' +
		'		<invert>0</invert>\n' +
		'		<item>\n' +
		'			<hlevel>' + cdata.level + '</hlevel>\n' +
		'			<item_key>' + cdata.key + '</item_key>\n' +
		'			<dim_tablename>' + cdata.tablename + '</dim_tablename>\n' +
		'			<dim_dimcode>' + cdata.dimcode + '</dim_dimcode>\n' +
		'			<item_is_synonym>' + cdata.synonym + '</item_is_synonym>\n' + 
		'		</item>\n' +
		'	</panel>\n';
	}

	var outputOptions = '';		
	outputOptions += '	<patient_set select="using_input_list" onlykeys="false"/>\n';
	outputOptions += '	<observation_set blob="false" onlykeys="false"/>\n';
	
	var messageFilter = '';
	messageFilter += '<input_list>\n';
	messageFilter += '	 <patient_list min="' + min + '" max="' + max + '">\n';
	messageFilter += '	   <patient_set_coll_id>' + i2b2.ExportXLS.model.prsRecord.sdxInfo.sdxKeyValue + '</patient_set_coll_id>\n';
	messageFilter += '	 </patient_list>\n';
	messageFilter += '</input_list>\n';
	messageFilter += '<filter_list>\n';
	messageFilter += filterList;
	messageFilter += '</filter_list>\n';
	messageFilter += '<output_option>\n';
	messageFilter += outputOptions;
	messageFilter += '</output_option>\n';

	return messageFilter;
}

i2b2.ExportXLS.ClearResults = function() 
{
	i2b2.ExportXLS.Patients = new Array();
	i2b2.ExportXLS.ObservationSets = {};	

	// Delete all concept code denotation / ontology path which have not been resolved yet (-> keep those that HAVE been resolved).
	for (conceptCode in i2b2.ExportXLS.ConceptCodeDenotations)
	{
		var displayValue = i2b2.ExportXLS.ConceptCodeDenotations[conceptCode];

		if (displayValue.substr(0, 10) == "Unresolved")
		{
			delete i2b2.ExportXLS.ConceptCodeDenotations[conceptCode];
			delete i2b2.ExportXLS.ConceptCodeOntologyPaths[conceptCode];
		}
	}
}

i2b2.ExportXLS.GetResults = function() 
{
	if (i2b2.ExportXLS.model.dirtyResultsData) 
	{
		var queryFilter = i2b2.ExportXLS.GetPDOQueryFilter(i2b2.ExportXLS.PDOQueryChunkLower, i2b2.ExportXLS.PDOQueryChunkUpper);  
		i2b2.ExportXLS.PDOQueryChunkCurrent++;
		
		// Callback processor
		var scopedCallback = new i2b2_scopedCallback();
		scopedCallback.scope = this;
		
		scopedCallback.callback = function(results) { return i2b2.ExportXLS.GetResultsCallback(results); }
		
		$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-directions")[0].hide();
		$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-finished")[0].hide();
		$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-export")[0].hide();
		
		if ( (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails) || (i2b2.ExportXLS.model.outputOptions.includeOntologyPath) )
		{		
			$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-working")[0].hide();
			$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-working-hard")[0].show();
		}
		else
		{
			$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-working")[0].show();
			$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-working-hard")[0].hide();
		}
		
		i2b2.ExportXLS.UpdateProgressMessage("Currently busy with: Retrieving data (chunk " + i2b2.ExportXLS.PDOQueryChunkCurrent + ").")

		// AJAX CALL USING THE EXISTING CRC CELL COMMUNICATOR
		i2b2.CRC.ajax.getPDO_fromInputList("Plugin:ExportXLS", {PDO_Request: queryFilter}, scopedCallback);
	}
		
}

i2b2.ExportXLS.GetAllConceptNamesFromOntology = function()
{
	// *********************
	// First step: Write all concept codes that have not been queried yet 
	//             as keys into dictionary and fill it with default value 
	//             "(Unresolved concept code: {$concept code})"
	// *********************
	
	for (panelName in i2b2.ExportXLS.ObservationSets)
	{
		var thisObservationSetDictionary = i2b2.ExportXLS.ObservationSets[panelName];
		var thisObservationSetDictionary_Observations = thisObservationSetDictionary["observations"];
		
		for (var j = 0; j < thisObservationSetDictionary_Observations.length; j++)
		{
			var thisObservation = thisObservationSetDictionary_Observations[j];		
			var thisObservationConceptCode = thisObservation["concept_cd"];
									
			if (  !Object.isUndefined(thisObservationConceptCode)  )
			{
				if (  Object.isUndefined(i2b2.ExportXLS.ConceptCodeDenotations[thisObservationConceptCode])  )
				{
					i2b2.ExportXLS.ConceptCodeDenotations[thisObservationConceptCode] = "Unresolved concept code: " + thisObservationConceptCode + "";
					i2b2.ExportXLS.ConceptCodeOntologyPaths[thisObservationConceptCode] = "";
				}
				else						
				{
					// For debugging only
					//alert("Skipping " + thisObservationConceptCode);			
				}
			}
		}
	
	}

	// *********************
	// Second step: Write all modifier codes that have not been queried yet 
	//              as keys into dictionary and fill it with default value 
	//              "(Unresolved modifier code: {$modifier code})"
	// *********************
	
	for (panelName in i2b2.ExportXLS.ObservationSets)
	{
		var thisObservationSetDictionary = i2b2.ExportXLS.ObservationSets[panelName];	
		var thisObservationSetDictionary_Observations = thisObservationSetDictionary["observations"];
		
		for (var j = 0; j < thisObservationSetDictionary_Observations.length; j++)
		{
			var thisObservation = thisObservationSetDictionary_Observations[j];		
			var thisObservationModifierCode = thisObservation["modifier_cd"];
												
			if (  !Object.isUndefined(thisObservationModifierCode)  )
			{
				if (  Object.isUndefined(i2b2.ExportXLS.ConceptCodeDenotations[thisObservationModifierCode])  )
				{
					i2b2.ExportXLS.ConceptCodeDenotations[thisObservationModifierCode] = "Unresolved modifier code: " + thisObservationModifierCode + "";
					i2b2.ExportXLS.ConceptCodeOntologyPaths[thisObservationModifierCode] = "";
				}
				else						
				{
					// For debugging only
					//alert("Skipping " + thisObservationModifierCode);			
				}
			}
		}
	
	}
	
	// *********************
	// Third step: start resolving the concept codes one after the other.
	// Cannot handle this in a loop since querying requires a callback function.
	// This callback function then starts a query for the next concept code until all codes have been resolved.
	// *********************
	
	i2b2.ExportXLS.GetNextConceptNamefromOntology();		
}

i2b2.ExportXLS.GetNextConceptNamefromOntology = function()
{
	// *********************
	// Resolving the next unresolved concept code.
	// If finished, the callback function then starts a new query for the next concept code until all codes have been resolved.
	// *********************
    
	var conceptCode = "";
	var allResolved = true;	
	
	var numberOfConcepts = Object.keys(i2b2.ExportXLS.ConceptCodeDenotations).length;
	var currentConceptNumber = 0;
	
	for (conceptCode in i2b2.ExportXLS.ConceptCodeDenotations)
	{
		var displayValue = i2b2.ExportXLS.ConceptCodeDenotations[conceptCode];
		currentConceptNumber++;
		
		if (displayValue.substr(0, 10) == "Unresolved")
		{
			allResolved = false;
			
			var conceptCodeElements = conceptCode.split(":");

			var codingSystem = conceptCodeElements[0];
			var codingTerm = "";

			if (conceptCodeElements.length > 1)
			{
				codingTerm = conceptCodeElements[1];
			}

			i2b2.ExportXLS.UpdateProgressMessage("Currently busy with: Querying concept details from ontology (" + currentConceptNumber + " of " + numberOfConcepts + ").")
			
			// The callback of this method sets the dictionary entry and then again calls this method.
			i2b2.ExportXLS.LaunchQueryToGetConceptNameFromOntology(codingSystem, codingTerm)

			break;
		}
	
	}

	if (allResolved)
	{
		i2b2.ExportXLS.AllConceptNamesRetrieved();		
	}
	
}

i2b2.ExportXLS.LaunchQueryToGetConceptNameFromOntology = function(codingSystem, codingTerm) 
{
	// Set Callback
	var scopedCallback = new i2b2_scopedCallback();
	scopedCallback.scope = this;
	scopedCallback.callback = function(results) { i2b2.ExportXLS.GetConceptNameFromOntologyCallback(codingSystem, codingTerm, results);	}
	
	// Set Options
	var searchOptions = {};
	searchOptions.ont_max_records = "max='" + i2b2.ONT.view['find'].params.max + "' ";
	searchOptions.ont_synonym_records = i2b2.ONT.view['find'].params.synonyms;
	searchOptions.ont_hidden_records = i2b2.ONT.view['find'].params.hiddens;
	searchOptions.ont_search_strategy = "exact";
	searchOptions.ont_search_coding = codingSystem + ":";
	searchOptions.ont_search_string = codingTerm;
	
	i2b2.ONT.ajax.GetCodeInfo("Plugin:ExportXLS", searchOptions, scopedCallback);
}

i2b2.ExportXLS.GetConceptNameFromOntologyCallback = function(codingSystem, codingTerm, resultData)
{
	var conceptCode = codingSystem;

	if (codingTerm != "")
	{
		conceptCode += ":" + codingTerm;
	}
	
	// THIS function is used to process the AJAX resultData of the getChild call
	//		resultData data object contains the following attributes:
	//			refXML: xmlDomObject <--- for data processing
	//			msgRequest: xml (string)
	//			msgResponse: xml (string)
	//			error: boolean
	//			errorStatus: string [only with error=true]
	//			errorMsg: string [only with error=true]
	
	// Check for errors
	if (i2b2.ExportXLS.QueryResultHasErrors(resultData)) 
	{
		return false;
	}

	var xmlResponse = i2b2.ExportXLS.GetXMLResponseFromQueryResult(resultData);
	var xmlDoc = xmlResponse["xmlDoc"];
	

	var conceptName = "";
	var conceptNameOfBranch = "";
	var conceptNameUnresolvable = "{Unresolvable code: '" + conceptCode + "'}";
	var conceptOnotlogyPath = "";
	
	$j(xmlDoc).find('concept').each( function() 
		{ 
			conceptNode = $j(this);
			
			var visualattributes = conceptNode.find("visualattributes").text();

			if (visualattributes == "FA ") 
			{
				conceptNameOfBranch = conceptNode.find("name").text();		
			}
			else
			{
				conceptName = conceptNode.find("name").text();
			}

			//if (visualattributes == "LA ") 
			//{
			//	conceptName = conceptNode.find("name").text();
			//}

			//if (visualattributes == "RA ") 
			//{
			//	conceptName = conceptNode.find("name").text();
			//}
			
			//if (visualattributes == "LI ") 
			//{
			//	conceptName = conceptNode.find("name").text();
			//}
			
			conceptOnotlogyPath = conceptNode.find("key").text();
		} 
	);
	
	if (conceptName == "")
	{
		if (conceptNameOfBranch != "")
		{
			conceptName = conceptNameOfBranch;
		}
		else
		{
			conceptName = conceptNameUnresolvable;
		}
	}
	
	
	i2b2.ExportXLS.ConceptCodeDenotations[conceptCode] = conceptName;
	i2b2.ExportXLS.ConceptCodeOntologyPaths[conceptCode] = conceptOnotlogyPath;
	
	i2b2.ExportXLS.GetNextConceptNamefromOntology();
}

i2b2.ExportXLS.AllConceptNamesRetrieved = function()
{
	i2b2.ExportXLS.FinishProcessing();	
}

i2b2.ExportXLS.ResolveConceptCode = function(observation)
{
	var conceptCode = observation["concept_cd"];
	var result = "";
	
	if (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails)
	{	
		result = i2b2.ExportXLS.ConceptCodeDenotations[conceptCode];
	}
	
	if (Object.isUndefined(result))  // This should not happen... just for development & debugging purposes!	
	{
		result = "";
	}

	return result;
}

i2b2.ExportXLS.GetConceptOntologyPath = function(observation)
{
	var conceptCode = observation["concept_cd"];
	var result = "";
		
	if (i2b2.ExportXLS.model.outputOptions.includeOntologyPath)
	{	
		result = i2b2.ExportXLS.ConceptCodeOntologyPaths[conceptCode];
	}
	
	if (Object.isUndefined(result))  // This should not happen... just for development & debugging purposes!	
	{
		result = "";
	}

	return result;
}

i2b2.ExportXLS.ResolveModifierCode = function(observation)
{
	var modifierCode = observation["modifier_cd"];	
	
	if ( (Object.isUndefined(modifierCode)) || (modifierCode == "@") )
	{
		modifierCode = "";
	}
	
	var result = modifierCode;
	
	if (modifierCode != "")
	{
		if (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails)
		{	
			var modifierDisplayValue = i2b2.ExportXLS.ConceptCodeDenotations[modifierCode];
			
			if (Object.isUndefined(modifierDisplayValue))
			{
				result = "";
			}
			else
			{
				result = modifierDisplayValue;
			}		
		}
	}
	
	if (Object.isUndefined(result))  // This should not happen... just for development & debugging purposes!	
	{
		result = modifierCode;
	}

	return result;
}

i2b2.ExportXLS.GetObservationSetDisplayName = function(key)
{
	var result = "";
	
	for (var i = 0; i < i2b2.ExportXLS.model.concepts.length; i++) {
		
		if (i2b2.ExportXLS.model.concepts[i].sdxInfo.sdxKeyValue == key) {
			result = i2b2.ExportXLS.model.concepts[i].sdxInfo.sdxDisplayName;
		}
	}
	
	return result;
}

i2b2.ExportXLS.ArraysAreEqual = function(templateArray, newArray)
{
	if (templateArray.length == newArray.length)
	{
		valuesAreEqual = true;
		
		for (var i = 0; i < templateArray.length; i++)
		{
			if (templateArray[i] != newArray[i])
			{
				valuesAreEqual = false;
				break;
			}
		}
		
		if (valuesAreEqual)
		{
			return true;
		}
	}

	return false;
}

i2b2.ExportXLS.CheckForDuplicates = function(arrayCollection, newArray) 	
{
	for (var i = 0; i < arrayCollection.length; i++)  // Loop through all arrays in the array collection
	{
		compareArray = arrayCollection[i];
		
		if (i2b2.ExportXLS.ArraysAreEqual(compareArray, newArray))
		{
			return true;
		}
	}
	
	return false;
}

i2b2.ExportXLS.ReplacePatientIDsByAscendingNumber = function() 	
{
	for (var i = 0; i < i2b2.ExportXLS.Patients.length; i++)
	{
		var thisPatient = i2b2.ExportXLS.Patients[i];
		thisPatient["patient_id_for_display"] = i + 1;		
	}		
}

i2b2.ExportXLS.FillPatientsDictionaryFromPatientNode = function(patientNode)
{
	var patient_id = patientNode.find("patient_id").text();

	var patientDataDictionary = {};

	patientDataDictionary["patient_id"] = patient_id;				
	patientDataDictionary["patient_id_for_display"] = patient_id;				
	patientDataDictionary["zip_cd"] = "";
	
	$j(patientNode).find('param').each(
		function() 
		{
			var paramName = $j(this).attr('column');
			var paramValue = $j(this).text();			
			patientDataDictionary[paramName] = paramValue;
			
			if (paramName == "birth_date")
			{
				patientDataDictionary["birth_date"] = i2b2.ExportXLS.GetFormattedDateTimeString(paramValue);
				patientDataDictionary["birth_year"] = i2b2.ExportXLS.GetFormattedYearString(paramValue);
			}

			if (paramName == "statecityzip_path")
			{
				i2b2.ExportXLS.GetStateCityZIPString(patientDataDictionary, paramValue);
			}

		}
	);		
	
	i2b2.ExportXLS.Patients.push(patientDataDictionary);
	
}

i2b2.ExportXLS.CheckIfServerPagedResult = function(xmlDoc) 	
{	
	var result = {};
	
	var pagingNode = $j(xmlDoc).find('paging_by_patients');
	
	var firstIndex = pagingNode.find('patients_returned').attr('first_index');
	var lastIndex = pagingNode.find('patients_returned').attr('last_index');

	if ( (!Object.isUndefined(firstIndex)) && (!Object.isUndefined(lastIndex)) )
	{
		result["firstIndex"] = parseInt(firstIndex);
		result["lastIndex"]  = parseInt(lastIndex);
		i2b2.ExportXLS.PDOQueryPagedByServer = true;
	}
	
	return result;
}

i2b2.ExportXLS.FillPatientsDictionary = function(xmlDoc) 	
{	
	$j(xmlDoc).find('patient').each( function() { i2b2.ExportXLS.FillPatientsDictionaryFromPatientNode($j(this)) } );
}	

i2b2.ExportXLS.FillObservationDataDictionary = function(observationNode, childNode, observationDataDictionary)
{
	var textValue = observationNode.find(childNode).text();
	
	if ( (textValue != "@") && (textValue != "")  )
	{	
		observationDataDictionary[childNode] = textValue;
	}	
}

i2b2.ExportXLS.GetObservationDataDictionaryFromObservationNode = function(observationNode)
{
	var observationDataDictionary = {};
	
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "patient_id", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "concept_cd", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "start_date", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "end_date", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "valuetype_cd", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "tval_char", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "nval_num", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "units_cd", observationDataDictionary)
	i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "modifier_cd", observationDataDictionary)
	
	// Not used in this version
	//i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "valueflag_cd", observationDataDictionary)
	//i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "location_cd", observationDataDictionary)
	//i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "instance_num", observationDataDictionary)
	//i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "event_id", observationDataDictionary)
	//i2b2.ExportXLS.FillObservationDataDictionary(observationNode, "observer_cd", observationDataDictionary)	
		
	return observationDataDictionary;
}

i2b2.ExportXLS.FillObservationSetsDictionaryFromObservationSetNode = function(observationSetNode)
{
	var panelName = observationSetNode.attr('panel_name');
	var displayName = i2b2.ExportXLS.GetObservationSetDisplayName(panelName);
	
	var thisObservationSetDictionary = i2b2.ExportXLS.ObservationSets[panelName];
	
	if (Object.isUndefined(thisObservationSetDictionary))
	{
		thisObservationSetDictionary = { "display_name": displayName, "panel_name": panelName, "observations": new Array() }
	}
	
	$j(observationSetNode).find('observation').each( function() { thisObservationSetDictionary["observations"].push(i2b2.ExportXLS.GetObservationDataDictionaryFromObservationNode($j(this))); } );
	
	i2b2.ExportXLS.ObservationSets[panelName] = thisObservationSetDictionary;
}

i2b2.ExportXLS.FillObservationSetsArray = function(xmlContent) 	
{	
	// Optimization idea / TODO: 
	// As first step loop through all observation sets to get panel names and put them into a dictionary. 
	// Then the name does not need to be solved every time again in GetObservationDataDictionaryFromObservationNode()
	
	$j(xmlContent).find('ns2\\:observation_set').each( function() { i2b2.ExportXLS.FillObservationSetsDictionaryFromObservationSetNode($j(this)) } );	
}

i2b2.ExportXLS.OutputFormatIs = function(format_string)
{
	return (i2b2.ExportXLS.model.outputOptions.outputFormat == format_string);
}

i2b2.ExportXLS.GetObservationStringForValue = function(observation)
{
	var result = "";

	if ((observation["valuetype_cd"]) == "N")
	{
		result = observation["nval_num"];
	}
	else if ((observation["valuetype_cd"]) == "T")
	{
		result = observation["tval_char"];
	}
	else if ((observation["valuetype_cd"]) == "B")
	{
		result = "[BLOB]";
	}
	
	return result;
}

i2b2.ExportXLS.GetObservationStringForUnit = function(observation)
{
	var unit = observation["units_cd"];	

	if (  (Object.isUndefined(unit)) || (unit == "@") || (unit == "undefined")  )
	{
		unit = "";
	}
	
	return unit;
}

i2b2.ExportXLS.GetObservationStringForAllComponents = function(observation)
{
	var result = "";
	
	var observationConcept = observation["concept_cd"];
	var observationConceptDenotation = i2b2.ExportXLS.ResolveConceptCode(observation);	
	var observationConceptOntologyPath = i2b2.ExportXLS.GetConceptOntologyPath(observation);
	var observationValue = i2b2.ExportXLS.GetObservationStringForValue(observation);
	var observationUnit = i2b2.ExportXLS.GetObservationStringForUnit(observation);
	var observationModifier = i2b2.ExportXLS.ResolveModifierCode(observation);
	
	var observationConceptWithPath = observationConcept;
	
	if ( (i2b2.ExportXLS.model.outputOptions.includeOntologyPath) && (observationConceptOntologyPath != "") )
	{
		observationConceptWithPath = observationConceptOntologyPath + observationConceptWithPath;
	}
	
	if (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails)
	{
		observationConceptWithPath = i2b2.ExportXLS.ResolveConceptCode(observation) + " [" + observationConceptWithPath + "]";
	}

	if (observationUnit != "")
	{
		observationUnit = " " + observationUnit;
	}

	var result = observationConceptWithPath;

	if (observationValue != "")
	{
		result += " = " + observationValue + observationUnit
	}

	if (observationModifier != "")
	{
		result += " (" + observationModifier + ")"
	}
	
	return result;
}

i2b2.ExportXLS.CreateResultMatrixHeaderLine = function()
{
	var headerLine = new Array();
	
	headerLine.push(" ");           		// Always add column # header 
	
	if (i2b2.ExportXLS.model.outputOptions.replacePatientID)
	{
		headerLine.push("Patient Number");	// Always add Patient Number header... 
	}
	else
	{
		headerLine.push("Patient ID");		// ...or Patient ID header
	}

	for (var i = 0; i < i2b2.ExportXLS.PatientDataColumns.length; i++)  // If no demographic data is wanted, array length is 0
	{
		headerLine.push(i2b2.ExportXLS.PatientDataColumnHeaders[i2b2.ExportXLS.PatientDataColumns[i]]);	
	}

	if ( (i2b2.ExportXLS.OutputFormatIs("single_filtered")) || (i2b2.ExportXLS.OutputFormatIs("aggregated")) )
	{
		for (panelName in i2b2.ExportXLS.ObservationSets)
		{
			var thisObservationSet = i2b2.ExportXLS.ObservationSets[panelName];
			i2b2.ExportXLS.ConceptColumns.push(thisObservationSet["panel_name"]);  // Store which concept is displayed in which column #		
			headerLine.push(thisObservationSet["display_name"]);	
		}	
	}	
	else if (i2b2.ExportXLS.OutputFormatIs("single_detailed")) 
	{
		headerLine.push("Timestamp Start");
		headerLine.push("Timestamp End");
		
		for (panelName in i2b2.ExportXLS.ObservationSets)
		{
			var thisObservationSet = i2b2.ExportXLS.ObservationSets[panelName];
			i2b2.ExportXLS.ConceptColumns.push(thisObservationSet["panel_name"]);  // Store which concept is displayed in which column #		
			headerLine.push(thisObservationSet["display_name"]);	
		}	
	}	
	else if (i2b2.ExportXLS.OutputFormatIs("single_raw")) 
	{
		headerLine.push("Timestamp Start");
		headerLine.push("Timestamp End");
		headerLine.push("Observation Set");

		if (i2b2.ExportXLS.model.outputOptions.includeOntologyPath)
		{
			headerLine.push("Observation Concept Ontology Path");
		}
		
		headerLine.push("Observation Concept Code");

		if (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails)
		{
			headerLine.push("Observation Concept Denotation");
		}
		
		headerLine.push("Observation Value");
		headerLine.push("Observation Unit");
		headerLine.push("Observation Modifier");	
	}	

	i2b2.ExportXLS.model.dirtyMatrixData = false;
	
	return headerLine;
}

i2b2.ExportXLS.GetFormattedYearString = function(dateTimeString)
{
	var year = "";
	
	if (!Object.isUndefined(dateTimeString))
	{
		if (dateTimeString.length >= 4)
		{
			year = dateTimeString.substr(0,4);
		}
	}
	
	return year
}

i2b2.ExportXLS.GetFormattedDateTimeString = function(dateTimeString)
{
	var result = "";
	
	var date = "";
	var time = "";
	
	if (!Object.isUndefined(dateTimeString))
	{
		if (dateTimeString.length >= 10)
		{
			date = dateTimeString.substr(0,10);
		}
		
		if (dateTimeString.length >= 18)
		{
			time = dateTimeString.substr(11,8);
		}
	}

	if (time == "00:00:00") 
	{
		time = "";
	}
	else if (time != "")
	{
		time = " " + time;
	}
	
	result = date + time;
	
	return result
}

i2b2.ExportXLS.GetStateCityZIPString = function(patientDictionary, value)
{
	// Value example: "Zip codes\Massachusetts\Northampton\01061\"
	var state = "";
	var city = "";
	var zip = "";
	
	// Remove leading "Zip codes" string if it exists
	if ("Zip code" == value.substring(0, 8)) 
	{
		value = value.substring(10, value.length);
	}
	
	var valueElements = value.split("\\");

	if (valueElements.length > 0)
	{
		state = valueElements[0];
	}

	if (valueElements.length > 1)
	{
		city = valueElements[1];
	}

	if (valueElements.length > 2)
	{
		zip = valueElements[2];
	}
	
	patientDictionary["statecityzip_path"] = state

	if (city != "")
	{
		if (patientDictionary["statecityzip_path"] != "")
		{
			patientDictionary["statecityzip_path"] += "/"
		}
		
		patientDictionary["statecityzip_path"] += city
	}

	if (zip != "")
	{
		if (patientDictionary["statecityzip_path"] != "")
		{
			patientDictionary["statecityzip_path"] += "/"
		}

		patientDictionary["statecityzip_path"] += zip
	}

	patientDictionary["state_path"] = state;
	patientDictionary["city_path"] = city;

	if (patientDictionary["zip_cd"] == "")  // Do not overwrite if already set by "zip_cd" param!
	{
		patientDictionary["zip_cd"] = zip;				
	}

}

i2b2.ExportXLS.AddDataLine_SingleFiltered = function(dataLines, patientID, patientDemographicData, observation, observationSetPanelName)
{
	var thisNewDataLine = new Array();
	thisNewDataLine.push(patientID);
	thisNewDataLine = thisNewDataLine.concat(patientDemographicData);

	for (var conceptColumnIndex = 0; conceptColumnIndex < i2b2.ExportXLS.ConceptColumns.length; conceptColumnIndex++)
	{
		if (i2b2.ExportXLS.ConceptColumns[conceptColumnIndex] == observationSetPanelName)
		{
			thisNewDataLine.push(i2b2.ExportXLS.GetObservationStringForAllComponents(observation));
		}
		else
		{
			thisNewDataLine.push("");
		}		
	}
	
	if (!i2b2.ExportXLS.CheckForDuplicates(dataLines,thisNewDataLine))  // This is a little tricky, because indexOf always returns -1, since arrays are always different instances, even if the values are the same!
	{
		dataLines.push(thisNewDataLine);
	}
}

i2b2.ExportXLS.AddDataLine_SingleDetailed = function(dataLines, patientID, patientDemographicData, observation, observationSetPanelName, j)
{
	var thisNewDataLine = new Array();
	thisNewDataLine.push(patientID);
	thisNewDataLine = thisNewDataLine.concat(patientDemographicData);
	
	thisNewDataLine.push(i2b2.ExportXLS.GetFormattedDateTimeString(observation["start_date"]));
	thisNewDataLine.push(i2b2.ExportXLS.GetFormattedDateTimeString(observation["end_date"]));
	
	for (var conceptColumnIndex = 0; conceptColumnIndex < i2b2.ExportXLS.ConceptColumns.length; conceptColumnIndex++)
	{
		if (i2b2.ExportXLS.ConceptColumns[conceptColumnIndex] == observationSetPanelName)
		{
			thisNewDataLine.push(i2b2.ExportXLS.GetObservationStringForAllComponents(observation));
		}
		else
		{
			thisNewDataLine.push("");
		}		
	}
	
	dataLines.push(thisNewDataLine);
}

i2b2.ExportXLS.AddDataLine_SingleRaw = function(dataLines, patientID, patientDemographicData, observation, observationSetDisplayName)
{
	var thisNewDataLine = new Array();
	thisNewDataLine.push(patientID);
	thisNewDataLine = thisNewDataLine.concat(patientDemographicData);
	
	thisNewDataLine.push(i2b2.ExportXLS.GetFormattedDateTimeString(observation["start_date"]));
	thisNewDataLine.push(i2b2.ExportXLS.GetFormattedDateTimeString(observation["end_date"]));
	
	thisNewDataLine.push(observationSetDisplayName);
	
	var observationConceptCode = observation["concept_cd"];
	var observationConceptDenotation = i2b2.ExportXLS.ResolveConceptCode(observation);
	var observationConceptOntologyPath = i2b2.ExportXLS.GetConceptOntologyPath(observation);
	var observationValue = i2b2.ExportXLS.GetObservationStringForValue(observation);
	var observationUnit = i2b2.ExportXLS.GetObservationStringForUnit(observation);
	var observationModifier = i2b2.ExportXLS.ResolveModifierCode(observation)
	
	if (i2b2.ExportXLS.model.outputOptions.includeOntologyPath)
	{
		thisNewDataLine.push(observationConceptOntologyPath);
	}
	
	thisNewDataLine.push(observationConceptCode);
	
	if (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails)
	{
		thisNewDataLine.push(observationConceptDenotation);
	}
	
	thisNewDataLine.push(observationValue);
	thisNewDataLine.push(observationUnit);
	thisNewDataLine.push(observationModifier);

	dataLines.push(thisNewDataLine);
}

i2b2.ExportXLS.CreateResultMatrixDataLines = function()
{
	var dataLines = new Array();
	
	var patientNumber = 1;
	var increasePatientNumber = false;
	
	for (var p = 0; p < i2b2.ExportXLS.Patients.length; p++) // First loop over all patients
	{  	
		var thisPatient = i2b2.ExportXLS.Patients[p];
		var thisPatientID = thisPatient["patient_id"];
		var thisPatientIDForDisplay = thisPatient["patient_id_for_display"];
		
		if (i2b2.ExportXLS.model.outputOptions.replacePatientID)
		{
			thisPatientIDForDisplay = patientNumber;
		}
				
		var dataLine_PatientDemographicData = new Array();
		
		for (var i = 0; i < i2b2.ExportXLS.PatientDataColumns.length; i++)   // If no demographic data is wanted, array length is 0
		{
			var patientDemoDataColumn = i2b2.ExportXLS.PatientDataColumns[i];
			var patientDemoDataValue = thisPatient[patientDemoDataColumn];
			
			if (patientDemoDataColumn == "birth_date")
			{
				patientDemoDataValue = i2b2.ExportXLS.GetFormattedDateTimeString(patientDemoDataValue);
			}
		
			dataLine_PatientDemographicData.push(patientDemoDataValue);	
		}
		
		if (i2b2.ExportXLS.OutputFormatIs("aggregated"))
		{
			var dataline_aggregated = new Array();
			dataline_aggregated.push(thisPatientIDForDisplay);
			dataline_aggregated = dataline_aggregated.concat(dataLine_PatientDemographicData);
			var aggregatedColumsDictionary = {};
			increasePatientNumber = true;
		}
		
		for (panelName in i2b2.ExportXLS.ObservationSets)  // Now loop over all observation sets
		{
			var thisObservationSet = i2b2.ExportXLS.ObservationSets[panelName];
			var thisObservationSetDisplayName = thisObservationSet["display_name"];
			var thisObservationSetPanelName = thisObservationSet["panel_name"];
			
			if (i2b2.ExportXLS.OutputFormatIs("aggregated"))
			{
				aggregatedObservationsArray = new Array();				
			}
			
			for (var j = 0; j < thisObservationSet["observations"].length; j++)  // And now loop over all observations of this set
			{				
				thisObservation = thisObservationSet["observations"][j];
				
				if (thisObservation["patient_id"] == thisPatientID)
				{
					if  (i2b2.ExportXLS.OutputFormatIs("single_filtered"))
					{
						i2b2.ExportXLS.AddDataLine_SingleFiltered(dataLines, thisPatientIDForDisplay, dataLine_PatientDemographicData, thisObservation, thisObservationSetPanelName);	
						increasePatientNumber = true;
					}
					else if (i2b2.ExportXLS.OutputFormatIs("single_detailed")) 
					{
						i2b2.ExportXLS.AddDataLine_SingleDetailed(dataLines, thisPatientIDForDisplay, dataLine_PatientDemographicData, thisObservation, thisObservationSetPanelName, j);
						increasePatientNumber = true;
					}
					else if (i2b2.ExportXLS.OutputFormatIs("single_raw")) 
					{
						i2b2.ExportXLS.AddDataLine_SingleRaw(dataLines, thisPatientIDForDisplay, dataLine_PatientDemographicData, thisObservation, thisObservationSetDisplayName);
						increasePatientNumber = true;
					}
					else if (i2b2.ExportXLS.OutputFormatIs("aggregated"))
					{
						observationString = i2b2.ExportXLS.GetObservationStringForAllComponents(thisObservation);
						
						if (-1 == aggregatedObservationsArray.indexOf(observationString))
						{					
							aggregatedObservationsArray.push(observationString);
						}
					}
					
				}
				
			}
			
			if (i2b2.ExportXLS.OutputFormatIs("aggregated"))
			{
				var aggregatedCell = "";
				
				for (var l = 0; l < aggregatedObservationsArray.length; l++)
				{
					aggregatedCell += aggregatedObservationsArray[l];
					
					if (l < aggregatedObservationsArray.length)
					{
						aggregatedCell += "\n";
					}
				}
			
				aggregatedColumsDictionary[thisObservationSetPanelName] = aggregatedCell;				
			}
			
		}

		if (i2b2.ExportXLS.OutputFormatIs("aggregated")) 
		{		
			var data_exists = false;  // This flag stores wheter ANY data exists for this patient. If not: do not add data row!
			
			var k = 0;
			for (panelName in i2b2.ExportXLS.ObservationSets)
			{
				var thisObservationSet = i2b2.ExportXLS.ObservationSets[panelName];
				var thisObservationSetPanelName = thisObservationSet["panel_name"];
				
				if (aggregatedColumsDictionary[thisObservationSetPanelName].length > 0)
				{
					data_exists = true;
				}
				
				// Find out the correct concept column #
				if (i2b2.ExportXLS.ConceptColumns[k] == thisObservationSetPanelName)
				{
					dataline_aggregated.push(aggregatedColumsDictionary[thisObservationSetPanelName]);
				}
				else
				{
					dataline_aggregated.push("");
				}
				
				k++;
			}	
		
			if (data_exists)
			{
				dataLines.push(dataline_aggregated);
			}
		}		

		if (increasePatientNumber)
		{
			patientNumber++;
			increasePatientNumber = false;
		}
	} 
	
	return dataLines;
}

i2b2.ExportXLS.CreatePatientDemoDataArray = function()
{
	i2b2.ExportXLS.PatientDataColumns = new Array ();
	
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Sex == true) 
	{
		i2b2.ExportXLS.PatientDataColumns.push("sex_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Age == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("age_in_years_num");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_BirthDate == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("birth_date");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_VitalStatus == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("vital_status_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Language == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("language_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Race == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("race_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Religion == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("religion_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_Income == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("income_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_StateCityZIP == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("statecityzip_path");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_State == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("state_path");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_City == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("city_path");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_ZIP == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("zip_cd");
	}
	if (i2b2.ExportXLS.model.outputOptions.includePatientDemoData_MaritalStatus == true)
	{
		i2b2.ExportXLS.PatientDataColumns.push("marital_status_cd");
	}
}

i2b2.ExportXLS.CreateResultMatrix = function()
{
	i2b2.ExportXLS.ResultMatrix = new Array();
	i2b2.ExportXLS.ConceptColumns = new Array();
	i2b2.ExportXLS.CreatePatientDemoDataArray();
	
	// **********************************************
	// Create colunmn headers
	// **********************************************

	var headerLine = i2b2.ExportXLS.CreateResultMatrixHeaderLine();
	i2b2.ExportXLS.ResultMatrix.push(headerLine);

	// **********************************************
	// Create data lines
	// **********************************************

	var dataLines = i2b2.ExportXLS.CreateResultMatrixDataLines()
	
	// **********************************************
	// Add row numbers to data lines
	//   Must be last step, because duplicates could 
	//   not be found otherwise (row numbers are 
	//   ALWAYS different)!
	// **********************************************
	
	for (var x = 0; x < dataLines.length; x++)
	{
		var dataline_with_rownumber = new Array();
		dataline_with_rownumber.push(x + 1);
		dataline_without_rownumber = dataLines[x];
		
		dataline_with_rownumber = dataline_with_rownumber.concat(dataLines[x]);
		
		i2b2.ExportXLS.ResultMatrix.push(dataline_with_rownumber);
	}
	
}

i2b2.ExportXLS.CreateHTMLTable = function()
{
	var html_table = "";
	
	html_table += "<table id=\"ReportTable\">\n";
	html_table += "<caption><b>Patient Information</b><br> for Patient Set <i>\'" + i2b2.ExportXLS.model.prsRecord.sdxInfo.sdxDisplayName + "\'<br>&nbsp;</i></caption>\n";
	
	// **********************
	// Add header row
	// **********************
	
	matrix_headerline = i2b2.ExportXLS.ResultMatrix[0];
	var table_header = "";
	table_header += "<tr>";
	
	for (var j = 0; j < matrix_headerline.length; j++)
	{
		table_header += "<th>" + matrix_headerline[j] + "</th>";
	}

	table_header += "</tr>\n";
	
	html_table += table_header;	
	
	// **********************
	// Add data rows
	// **********************
	
	for (var i = 1; i < i2b2.ExportXLS.ResultMatrix.length; i++)
	{
		matrix_dataline = i2b2.ExportXLS.ResultMatrix[i];
		var table_row = "";
		table_row += "<tr>";
		
		if (i2b2.ExportXLS.OutputFormatIs("aggregated"))
		{
			for (var j = 0; j < matrix_dataline.length; j++)
			{
				table_row += '<td align="center">' + matrix_dataline[j] + '</td>';  // The align="center" is for the exported HTML only. In the web client it is defined by CSS
				table_row = table_row.replace(/\n/g, '<br>');
			}
		}
		else
		{
			for (var j = 0; j < matrix_dataline.length; j++)
			{
				table_row += '<td align="center">' + matrix_dataline[j] + '</td>';  // The align="center" is for the exported HTML only. In the web client it is defined by CSS
			}
		}
		
		table_row += "</tr>";
		
		html_table += table_row + "\n";
	}
	
    // Close table	
	html_table += "</table>";	
	
	i2b2.ExportXLS.HTMLResult = html_table;	
}

i2b2.ExportXLS.CreateCSV = function()
{
	var csv = "";

	for (var i = 0; i < i2b2.ExportXLS.ResultMatrix.length; i++)
	{
		matrix_row = i2b2.ExportXLS.ResultMatrix[i];
		
		var csv_line = "";
		
		var cellDelimiter = '"';
		
		if ( ((i2b2.ExportXLS.model.outputOptions.excludeDelimiter)) && (!i2b2.ExportXLS.OutputFormatIs("aggregated")) ) 
		{
			cellDelimiter = "";
		}
		
		for (var j = 0; j < matrix_row.length; j++)
		{
			var cell = matrix_row[j];
			csv_line += cellDelimiter + cell + cellDelimiter + ';';
		}
	
		csv_line = csv_line.replace(/;$/, '');
		csv += csv_line + "\n";	
	}
	
	csv = csv.replace(/\n$/, '');	
	
	i2b2.ExportXLS.CSVExport = csv;	
}

i2b2.ExportXLS.SortPatientArrayByPatientID = function(resultData)
{
	i2b2.ExportXLS.Patients.sort(function() { return arguments[0]["patient_id"] > arguments[1]["patient_id"] } ); 

	return;
}

i2b2.ExportXLS.GetResultsCallback = function(resultData)
{
	// THIS function is used to process the AJAX resultData of the getChild call
	//		resultData data object contains the following attributes:
	//			refXML: xmlDomObject <--- for data processing
	//			msgRequest: xml (string)
	//			msgResponse: xml (string)
	//			error: boolean
	//			errorStatus: string [only with error=true]
	//			errorMsg: string [only with error=true]
	
	// Check for errors
	if (i2b2.ExportXLS.QueryResultHasErrors(resultData)) 
	{
		return false;
	}
	
	var xmlResponse = i2b2.ExportXLS.GetXMLResponseFromQueryResult(resultData);
	var xmlDoc = xmlResponse["xmlDoc"];
	var xmlContent = xmlResponse["xmlContent"];
	
	var pagingResult = i2b2.ExportXLS.CheckIfServerPagedResult(xmlDoc);
	
	if ( (!Object.isUndefined(pagingResult["firstIndex"])) && (!Object.isUndefined(pagingResult["lastIndex"])) )
	{
		i2b2.ExportXLS.PDOQueryChunkUpper = pagingResult["lastIndex"];
	}
	
	i2b2.ExportXLS.FillPatientsDictionary(xmlDoc);
	i2b2.ExportXLS.FillObservationSetsArray(xmlContent);
	
	if (i2b2.ExportXLS.AllChunksQueried())
	{
		i2b2.ExportXLS.model.dirtyResultsData = false;	
		
		i2b2.ExportXLS.SortPatientArrayByPatientID();
	
		if ( (i2b2.ExportXLS.model.outputOptions.resolveConceptDetails) || (i2b2.ExportXLS.model.outputOptions.includeOntologyPath) )
		{
			i2b2.ExportXLS.UpdateProgressMessage("Currently busy with: Querying concept details from ontology.")
			
			// This launches a method with a callback. 
			// The rest of the processing is handled in i2b2.ExportXLS.FinishProcessing() which is called by the callback.
			i2b2.ExportXLS.GetAllConceptNamesFromOntology();  
		}
		else
		{
			i2b2.ExportXLS.FinishProcessing();
		}
	}
	else
	{
		i2b2.ExportXLS.PDOQueryChunkLower = i2b2.ExportXLS.PDOQueryChunkUpper + 1;
		i2b2.ExportXLS.CalcPDOChunkUpper();
		
		i2b2.ExportXLS.GetResults(); 
	}
}

i2b2.ExportXLS.FinishProcessing = function()
{
	i2b2.ExportXLS.UpdateProgressMessage("Currently busy with: Creating table.")
	
	if ( i2b2.ExportXLS.PDOQueryPagedByServer )
	{
		alert("Warning!\n\nThe query has been paged by the server.\n\nYou can speed up further queries by setting or reducing\nthe 'Query Page Size' parameter on the 'Settings' tab.");
		i2b2.ExportXLS.PDOQueryPagedByServer = false;
	}

    i2b2.ExportXLS.CreateResultMatrix();
    i2b2.ExportXLS.CreateCSV();
    i2b2.ExportXLS.CreateHTMLTable();

	$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-working")[0].hide();
	$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-working-hard")[0].hide();
	$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-finished")[0].show();

	var divResults = $$("DIV#ExportXLS-mainDiv DIV#ExportXLS-InfoPDO")[0];			
	Element.select(divResults, '.InfoPDO-Response. originalXML')[0].innerHTML = '<pre>' + i2b2.ExportXLS.HTMLResult + '</pre>';	// Do not remove space character in '.InfoPDO-Response. originalXML'!

	$$("DIV#ExportXLS-mainDiv DIV#ExportXLS-TABS DIV.results-export")[0].show();
}
